<script>
    $(document).ready(function () {
        'use strict';

        const accountInfo = {};
        const siteCurrency = "{{ siteCurrency() }}";

        // Utility: Debounce function to limit API calls
        const debounce = (func, delay) => {
            let timer;
            return (...args) => {
                clearTimeout(timer);
                timer = setTimeout(() => func.apply(null, args), delay);
            };
        };

        // Fetch and populate withdrawal accounts
        const fetchWithdrawAccounts = debounce(walletId => {
            if (!walletId) return;

            const url = `${"{{ route('user.withdraw.account.index') }}"}?wallet_id=${encodeURIComponent(walletId)}`;

            $.ajax({
                url,
                type: "GET",
                success: response => updateWithdrawAccountOptions(response),
                error: () => updateWithdrawAccountOptions([], true)
            });
        }, 300);

		// Fetch withdrawal accounts on page load
		const walletSelect = $('.wallet-select');
        if (walletSelect.val() !== null) {
            fetchWithdrawAccounts(walletSelect.val());
        }
	    
        // Update withdrawal account dropdown
        const updateWithdrawAccountOptions = (accounts, isError = false) => {
            const $accountSelect = $('.withdraw-account-select');
            if (isError || !Array.isArray(accounts) || !accounts.length) {
                $accountSelect.html('<option disabled selected>Error or no accounts available</option>');
                return;
            }
            const options = accounts.map(account =>
                `<option value="${account.id}">${capitalize(account.name)}</option>`
            ).join('');
            $accountSelect.html('<option disabled selected>Select Account</option>' + options);
        };

        // Fetch account details
        const fetchAccountInfo = accountId => {
            if (!accountId) return;

            const url = "{{ route('user.withdraw.account.info', ':id') }}".replace(':id', encodeURIComponent(accountId));

            $.ajax({
                url,
                type: "GET",
                success: response => updateAccountInfo(response),
                error: () => showError('.account-info', 'Error loading account info')
            });
        };

        // Update account details UI
        const updateAccountInfo = data => {
            if (!data) return;
            const chargeType = data.charge_type === 'fixed' ? siteCurrency : '%';
            $('.account-info').text(`Charge: ${data.charge.toFixed()} ${chargeType}`);
            Object.assign(accountInfo, data);
        };

        // Update withdrawal summary
        const updateSummary = () => {
            const amount = parseFloat($('.amount-input').val()) || 0;
            const {
                processing_time,
                charge,
                charge_type,
                conversion_rate,
                currency,
                min_limit,
                max_limit
            } = accountInfo;

            if (!conversion_rate || !currency || !charge) return;

            // Validate amount
            if (amount < min_limit || amount > max_limit) {
                showError('.withdraw-amount-info', `Amount must be between ${min_limit} ${siteCurrency} and ${max_limit} ${siteCurrency}`);
                return;
            } else {
                showSuccess('.withdraw-amount-info', `Min: ${min_limit} ${siteCurrency} | Max: ${max_limit} ${siteCurrency}`);
            }

            // Calculate and display values
            const fee = charge_type === 'fixed' ? charge : (amount * charge) / 100;
            const total = amount + fee;
            const receivedAmount = amount * conversion_rate;
            const myWalletDecreased = total * conversion_rate;

            updateSummaryUI({
                processing_time,
                amount,
                fee,
                total,
                conversion_rate,
                receivedAmount,
                myWalletDecreased,
                currency
            });
        };

        // Update summary UI
        const updateSummaryUI = ({
                                     processing_time,
                                     amount,
                                     fee,
                                     total,
                                     conversion_rate,
                                     receivedAmount,
                                     myWalletDecreased,
                                     currency
                                 }) => {
            $('.processing-time').text(`${processing_time}`);
            $('.summary-amount').text(`${amount.toFixed(2)} ${siteCurrency}`);
            $('.summary-charge').text(`${fee.toFixed(2)} ${siteCurrency}`);
            $('.summary-total').text(`${total.toFixed(2)} ${siteCurrency}`);
            $('.conversion-rate').text(`1 ${siteCurrency} = ${conversion_rate} ${currency}`);
            $('.received-withdraw-amount').text(`${receivedAmount.toFixed(2)} ${currency}`);
            $('.my-wallet-decreased').text(`${myWalletDecreased.toFixed(2)} ${currency}`);
        };

        // Show error message
        const showError = (selector, message) => {
            $(selector).text(message).addClass('text-danger').removeClass('text-success');
        };

        // Show success message
        const showSuccess = (selector, message) => {
            $(selector).text(message).addClass('text-success').removeClass('text-danger');
        };

        // Utility: Capitalize string
        const capitalize = str => str.charAt(0).toUpperCase() + str.slice(1);

        // Event bindings
        $(document).on('change', '.wallet-select', function () {
            const walletId = this.value;
            fetchWithdrawAccounts(walletId);
        });

        $(document).on('change', '.withdraw-account-select', function () {
            const accountId = this.value;
            fetchAccountInfo(accountId);
        });

        $(document).on('input', '.amount-input', updateSummary);
    });
</script>
