<?php

namespace App\Models;

use App\Enums\MethodType;
use App\Services\CurrencyConversionService;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

class WithdrawMethod extends Model
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'withdraw_methods';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'payment_gateway_id',
        'logo',
        'name',
        'type',
        'method_code',
        'currency',
        'currency_symbol',
        'min_withdraw',
        'max_withdraw',
        'conversion_rate_live',
        'conversion_rate',
        'charge_type',
        'charge',
        'process_time_value',
        'process_time_unit',
        'fields',
        'status',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'type'               => MethodType::class,
        'min_withdraw'       => 'double',
        'max_withdraw'       => 'double',
        'rate'               => 'double',
        'charge'             => 'double',
        'process_time_value' => 'integer',
        'fields'             => 'array',
        'status'             => 'boolean',
    ];

    public function paymentGateway()
    {
        return $this->belongsTo(PaymentGateway::class, 'payment_gateway_id');
    }

    /**
     * Scope a query to only include active withdrawal methods.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    public function getConversionRateAttribute(): string
    {
        // Check if the conversion rate is set to live.
        if ($this->attributes['conversion_rate_live'] && $this->attributes['type'] === 'auto') {
            // Get an instance of the currency conversion service.
            $currencyConversionService = new CurrencyConversionService;

            // Convert the currency from the site's currency to the deposit method's currency.
            // The conversion rate is returned as a string with two decimal places.
            return number_format($currencyConversionService->convertCurrency(1, siteCurrency(), $this->attributes['currency']), 2, '.', '');
        }

        return $this->attributes['conversion_rate'];
    }

    /**
     * Get the full processing time as a string.
     */
    public function getProcessingTimeAttribute(): string
    {
        return $this->process_time_value ? "{$this->process_time_value} {$this->process_time_unit}" : __('Automated ASAP');
    }

    /**
     * Calculate the total charge for a given amount.
     */
    public function calculateCharge(float $amount): float
    {
        if ($this->charge_type === 'fixed') {
            return $this->charge;
        }

        if ($this->charge_type === 'percent') {
            return ($this->charge / 100) * $amount;
        }

        return 0.0;
    }

    /**
     * Check if a given amount is within the allowed limits.
     */
    public function isWithinLimits(float $amount): bool
    {
        return $amount >= $this->min_withdraw && $amount <= $this->max_withdraw;
    }

    public function getLogoAttribute(): string
    {
        // Check if the 'icon' attribute is null.
        if ($this->attributes['logo'] === null) {
            // If it is null, return the 'logo' attribute of the payment gateway.
            return $this->paymentGateway->logo;
        }

        // If the 'icon' attribute is not null, return it.
        return $this->attributes['logo'];
    }
}
