<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\MerchantStatus;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Merchant;
use App\Notifications\TemplateNotification;
use App\Traits\FileManageTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Notification;

class MerchantController extends Controller
{
    use FileManageTrait;

    public function index()
    {
        $merchants = Merchant::where('user_id', auth()->id())->get();

        return view('frontend.user.merchant.index', compact('merchants'));
    }

    public function create()
    {
        $currencies = auth()->user()->activeWallets()->pluck('currency');

        return view('frontend.user.merchant.create', compact('currencies'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'business_name'        => 'required|string|max:255',
            'site_url'             => 'nullable|url',
            'currency_id'          => 'required|exists:currencies,id',
            'business_logo'        => 'nullable|image|max:1024',
            'business_email'       => 'nullable|email|max:255',
            'business_description' => 'nullable|string|max:500',
        ]);

        // Attach the authenticated user's ID
        $validated['user_id'] = auth()->id();

        // Handle file upload if a business logo is provided
        if ($request->hasFile('business_logo')) {
            $validated['business_logo'] = $this->uploadImage($request->file('business_logo'));
        }

        // Use mass assignment to create the merchant
        Merchant::create($validated);

        // Notify admin
        $admins = Admin::permission('merchant-request-notification')->get();
        Notification::send($admins, new TemplateNotification(
            identifier: 'merchant_admin_notify_shop_request',
            data: [
                'user'           => auth()->user()->name,
                'business_name'  => $validated['business_name'],
                'business_email' => $validated['business_email'],
                'site_url'       => $validated['site_url'],
            ],
            sender: auth()->user(),
            action: route('admin.merchant.pending')
        ));

        // Send a success notification
        notifyEvs('success', __('New Merchant Request Successfully'));

        // Redirect to merchant index
        return to_route('user.merchant.index');
    }

    public function edit($id)
    {
        $currencies = auth()->user()->activeWallets()->pluck('currency');
        $merchant   = Merchant::findOrFail($id);

        return view('frontend.user.merchant.edit', compact('merchant', 'currencies'));
    }

    public function update(Request $request, $id)
    {
        $merchant = Merchant::findOrFail($id);

        $validated = $request->validate([
            'business_name'        => 'required|string|max:255',
            'site_url'             => 'nullable|url',
            'currency_id'          => 'required|exists:currencies,id',
            'business_logo'        => 'nullable|image|max:1024',
            'business_email'       => 'nullable|email|max:255',
            'business_description' => 'nullable|string|max:500',
        ]);

        // Check if currency has changed
        if ($merchant->currency_id !== (int) $validated['currency_id']) {
            $validated['status'] = MerchantStatus::PENDING;
        }

        // Handle file upload if a new business logo is provided
        if ($request->hasFile('business_logo')) {

            $validated['business_logo'] = $this->uploadImage($request->file('business_logo'), $merchant->business_logo);
        }

        // Update the merchant with validated data
        $merchant->update($validated);

        // Send a success notification
        notifyEvs('success', __('Merchant details updated successfully'));

        // Redirect to the merchant index page
        return to_route('user.merchant.index');
    }

    public function merchantConfig($id)
    {
        $merchant = Merchant::findOrFail($id);

        return view('frontend.user.merchant.config', compact('merchant'));
    }
}
