<?php

namespace App\Http\Controllers\Backend;

use App\Enums\UserStatus;
use App\Events\TransactionUpdated;
use App\Models\User;
use App\Models\UserFeature;
use App\Traits\FileManageTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends BaseController
{
    use FileManageTrait;

    public static function permissions(): array
    {
        return [
            'index|activeUser|suspendedUser|unverifiedUser' => 'user-list',
            'store'                                         => 'user-create',
            'destroy'                                       => 'user-delete',
        ];
    }

    /**
     * Display a listing of the users with filters.
     */
    public function index(Request $request)
    {

        $title = __('Users List');
        $users = User::query()->filter($request)->latest()->paginate(10)->withQueryString();

        return view('backend.user.index', compact('users', 'title'));
    }

    /**
     * Show active users.
     */
    public function activeUser(Request $request)
    {
        $title = __('Active Users');
        $users = User::query()->filter($request)->where('status', UserStatus::ACTIVE)->latest()->paginate(10)->withQueryString();

        $statusFilter = false;

        return view('backend.user.index', compact('users', 'title', 'statusFilter'));
    }

    public function suspendedUser(Request $request)
    {
        $title = __('Suspended Users');

        $users = User::query()->filter($request)->where('status', UserStatus::INACTIVE)->latest()->paginate(10)->withQueryString();

        return view('backend.user.index', compact('users', 'title'));
    }

    public function unverifiedUser(Request $request)
    {
        $title = __('Unverified Users');
        $users = User::query()->whereNull('email_verified_at')->filter($request)->latest()->paginate(10)->withQueryString();

        return view('backend.user.index', compact('users', 'title'));
    }

    public function kycUnverifiedUser(Request $request)
    {
        $title = __('KYC Unverified Users');
        $users = User::query()->kycUnverified()->filter($request)->latest()->paginate(10)->withQueryString();

        return view('backend.user.index', compact('users', 'title'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Step 1: Validate incoming request
        $validated = $request->validate([
            'avatar'     => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'first_name' => 'required|string|max:255',
            'last_name'  => 'required|string|max:255',
            'username'   => 'required|string|max:255|unique:users,username',
            'country'    => 'required|string',
            'phone'      => 'required|string',
            'email'      => 'required|email|unique:users,email',
            'password'   => 'required|same:confirm_password',
            'status'     => 'nullable|boolean',
            'role'       => 'required|string',
        ]);

        // Step 2: Process phone number
        [$countryName, $countryCode] = explode(':', $validated['country']);
        $formattedPhone              = formatPhone($countryCode, $validated['phone']);

        // Step 3: Handle avatar upload if provided
        $avatarPath = $request->hasFile('avatar')
            ? $this->uploadImage($request->file('avatar'))
            : null;

        // Step 4: Create user with transformed data
        $user = User::create([
            'avatar'     => $avatarPath,
            'first_name' => $validated['first_name'],
            'last_name'  => $validated['last_name'],
            'username'   => $validated['username'],
            'country'    => $countryName,
            'phone'      => $formattedPhone,
            'email'      => $validated['email'],
            'password'   => Hash::make($validated['password']),
            'role'       => $validated['role'],
            'status'     => $request->boolean('status') ? UserStatus::ACTIVE : UserStatus::INACTIVE,
        ]);

        // Step 5: Sync features and fire event
        UserFeature::syncWithConfigForUser($user->id);
        event(new TransactionUpdated($user));

        // Step 6: Notify and redirect
        notifyEvs('success', __('User created successfully'));

        return redirect()->route('admin.user.index');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {

        User::destroy($id);
        notifyEvs('success', __('User deleted successfully'));

        return redirect()->route('admin.user.index');
    }
}
