<?php

namespace App\Http\Controllers\Backend\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Contracts\Auth\Guard;
use Illuminate\Contracts\Auth\StatefulGuard;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Support\Facades\Auth;

class AuthController extends Controller implements HasMiddleware
{
    /**
     * This function returns an array of middleware for the AuthController.
     *
     * The middleware ensures that only guests can access all routes except the 'logout' route.
     *
     * @return array The middleware array.
     */
    public static function middleware(): array
    {
        // Create a new instance of the Middleware class with the 'guest:admin' role
        // and the 'except' option set to ['logout'].
        return [
            new Middleware('guest:admin', except: ['logout']),
        ];
    }

    /**
     * @return Application|Factory|View
     */
    public function loginView()
    {
        return view('backend.auth.login');
    }

    /**
     * Handle an authentication attempt.
     *
     * @return RedirectResponse
     */
    public function authenticate(Request $request)
    {
        $credentials = $request->validate([
            'email'    => ['required', 'email'],
            'password' => ['required'],
        ]);

        // Attempt to authenticate the user
        if ($this->guard()->attempt($credentials, $request->boolean('remember'))) {
            notifyEvs('success', 'Successfully login your account 🔥 !');

            return redirect()->intended('admin');
        }
        notifyEvs('warning', 'The provided credentials do not match our records.. ⚡️');

        return back();
    }

    /**
     * @return RedirectResponse
     */
    public function logout(Request $request)
    {
        $this->guard()->logout();

        // Clear session completely
        $request->session()->invalidate();

        // Regenerate token to prevent session fixation
        $request->session()->regenerateToken();

        // Remove cached admin permission
        $request->session()->forget('admin_permissions');

        return redirect()->route('admin.login');
    }

    /**
     * @return Guard|StatefulGuard
     */
    protected function guard()
    {
        return Auth::guard('admin');
    }
}
